# =========================== choose file dialog ===============================
function Get-FileName {
  param(
    [Parameter(Mandatory)][string]$WindowTitle,
    [Parameter(Mandatory)][string]$InitialDirectory,
    [string]$Filter = "All files (*.*)|*.*",
    [switch]$AllowMultiSelect
  )

  Add-Type -AssemblyName System.Windows.Forms
  $dialog = New-Object System.Windows.Forms.OpenFileDialog
  $dialog.Title = $WindowTitle
  $dialog.InitialDirectory = $InitialDirectory
  $dialog.Filter = $Filter
  $dialog.MultiSelect = $AllowMultiSelect
  $dialog.ShowHelp = $true
  $dialog.ShowDialog() | Out-Null

  if ($AllowMultiSelect) { return $dialog.Filenames } else { return $dialog.Filename }
}

# =========================== import ===========================================

# decimal format
$culture = [System.Globalization.CultureInfo]::CurrentCulture.Clone()
$culture.NumberFormat.NumberDecimalSeparator = ","
[System.Threading.Thread]::CurrentThread.CurrentCulture = $culture

# select file
if ($args[0]) {
  $fileName = $args[0]
} else {
  $fileName = Get-FileName -WindowTitle "Select file" -InitialDirectory . -Filter "Excel (*.xlsx) | *.xlsx"
}
if ($fileName -eq "") {
  Return
}

$fileNameOut = [System.IO.Path]::GetFileNameWithoutExtension($fileName) + "_out.csv"

# get account currency
$sheetName = (Get-ExcelSheetInfo -Path $fileName | Where-Object { $_.Name -like "Account*" })[0].Name
if ([string]::IsNullOrEmpty($sheetName)) {
  Return
}
$accountCurrency = (Import-Excel -Path $fileName -WorksheetName $sheetName -NoHeader -StartRow 1).P1[0].split(",")[1].Trim()

# import Excel trades
$sheetName = (Get-ExcelSheetInfo -Path $fileName | Where-Object { $_.Name -like "Trades*" }).Name
if ([string]::IsNullOrEmpty($sheetName)) {
  Return
}
$trades = (Import-Excel -Path $fileName -WorksheetName $sheetName -AsDate " Trade date ") |

# rename columns
Select-Object @{expression={$_.' Ticker '}; label='Ticker symbol'},
  @{expression={$_.' ISIN '}; label='ISIN'},
  @{expression={$_.' Direction '.Trim()}; label='Type'},
  @{expression={[double]$_.' Quantity '}; label='Shares'},
  @{expression={[double]$_.' Price '}; label='Gross Amount'},
  @{expression={$_.' Currency '}; label='Transaction Currency'},
  @{expression={$_.' Currency '}; label='Currency Gross Amount'},
  @{expression={[double]$_.' Amount '}; label='Value'},
  @{expression={[double]$_.' Fee '}; label='Fees'}, 
  ' Commission currency ',
  @{expression={$_.' Trade date '}; label='DateTime'},
  @{expression={$_.' Order ID '}; label='ID'} |
foreach {
  # add date
  $_ | Add-Member -MemberType NoteProperty -Name Date -Value $_.DateTime.split(" ")[0]
  # add time
  $_ | Add-Member -MemberType NoteProperty -Name Time -Value $_.DateTime.split(" ")[1]

  # add cash account (EUR, USD)
  $Currency = $_.'Transaction Currency'
  $_ | Add-Member -MemberType NoteProperty -Name "Cash Account" -Value "F24 $Currency"
  # add securities account (F24 EUR, F24 USD ...)
  if ([string]::IsNullOrEmpty($Currency)) {
    $_ | Add-Member -MemberType NoteProperty -Name "Securities Account" -Value ""
  } else {
    $_ | Add-Member -MemberType NoteProperty -Name "Securities Account" -Value "F24 $Currency"
  }

  # currency exchange
  $_ | Add-Member -MemberType NoteProperty -Name "Offset Account" -Value ""
  if ($_.'ISIN' -eq "-") {
    $_.Type = "Transfer (Outbound)"
    $_.'Gross Amount' = $_.Value
    $_.Value = $_.Shares
    $_.'Offset Account' = $_.'Cash Account'
    $_.'Securities Account' = ""
    $_.'Transaction Currency' = $_.'Ticker symbol'.split("/")[0]
    $_.'Cash Account' = "F24 " + $_.'Transaction Currency'
  }

  # export item
  $_
}

# import Excel cash
$sheetName = (Get-ExcelSheetInfo -Path $fileName | Where-Object { $_.Name -like "Cash In Out*" }).Name
if ([string]::IsNullOrEmpty($sheetName)) {
  Return
}
$cash = (Import-Excel -Path $fileName -WorksheetName $sheetName -AsDate " Trade date ") |

# rename columns
Select-Object @{expression={""}; label='Ticker symbol'},
  @{expression={""}; label='ISIN'},
  @{expression={$_.' Type '}; label='Type'},
  @{expression={0}; label='Shares'},
  @{expression={0}; label='Gross Amount'},
  @{expression={$_.' Currency '}; label='Transaction Currency'},
  @{expression={$_.' Currency '}; label='Currency Gross Amount'},
  @{expression={[double]$_.' Amount '}; label='Value'},
  @{expression={0}; label='Fees'},
  @{expression={$_.' Currency '}; label=' Commission currency '},
  @{expression={$_.' Date '}; label='DateTime'},
  @{expression={""}; label='ID'} |
foreach {
  # skip non-bank transfers
  if (!$_.Type.Contains("Bank transfer")) {
    return
  }
  if ($_.Value -ge 0) {
    $_.Type = "Deposit"
  } else {
    $_.Type = "Removal"
  }

  # add date
  $_ | Add-Member -MemberType NoteProperty -Name Date -Value $_.DateTime
  # add time
  $_ | Add-Member -MemberType NoteProperty -Name Time -Value "00:00:00"

  # add cash account (EUR, USD)
  $Currency = $_.'Transaction Currency'
  $_ | Add-Member -MemberType NoteProperty -Name "Cash Account" -Value "F24 $Currency"
  # add securities account (F24 EUR, F24 USD ...)
  if ([string]::IsNullOrEmpty($Currency)) {
    $_ | Add-Member -MemberType NoteProperty -Name "Securities Account" -Value ""
  } else {
    $_ | Add-Member -MemberType NoteProperty -Name "Securities Account" -Value "F24 $accountCurrency"
  }

  $_ | Add-Member -MemberType NoteProperty -Name "Offset Account" -Value ""

  # export item
  $_
}
 
($trades + $cash) | Export-Csv $fileNameOut -Delimiter ";" -Encoding UTF8 -NoTypeInformation