# =========================== choose file dialog ===============================
function Get-FileName
{
<#
.SYNOPSIS
   Show an Open File Dialog and return the file selected by the user

.DESCRIPTION
   Show an Open File Dialog and return the file selected by the user

.PARAMETER WindowTitle
   Message Box title
   Mandatory - [String]

.PARAMETER InitialDirectory
   Initial Directory for browsing
   Mandatory - [string]

.PARAMETER Filter
   Filter to apply
   Optional - [string]

.PARAMETER AllowMultiSelect
   Allow multi file selection
   Optional - switch

 .EXAMPLE
   Get-FileName
    cmdlet Get-FileName at position 1 of the command pipeline
    Provide values for the following parameters:
    WindowTitle: My Dialog Box
    InitialDirectory: c:\temp
    C:\Temp\42258.txt

    No passthru paramater then function requires the mandatory parameters (WindowsTitle and InitialDirectory)

.EXAMPLE
   Get-FileName -WindowTitle MyDialogBox -InitialDirectory c:\temp
   C:\Temp\41553.txt

   Choose only one file. All files extensions are allowed

.EXAMPLE
   Get-FileName -WindowTitle MyDialogBox -InitialDirectory c:\temp -AllowMultiSelect
   C:\Temp\8544.txt
   C:\Temp\42258.txt

   Choose multiple files. All files are allowed

.EXAMPLE
   Get-FileName -WindowTitle MyDialogBox -InitialDirectory c:\temp -AllowMultiSelect -Filter "text file (*.txt) | *.txt"
   C:\Temp\AES_PASSWORD_FILE.txt

   Choose multiple files but only one specific extension (here : .txt) is allowed

.EXAMPLE
   Get-FileName -WindowTitle MyDialogBox -InitialDirectory c:\temp -AllowMultiSelect -Filter "Text files (*.txt)|*.txt| csv files (*.csv)|*.csv | log files (*.log) | *.log"
   C:\Temp\logrobo.log
   C:\Temp\mylogfile.log

   Choose multiple file with the same extension

.EXAMPLE
   Get-FileName -WindowTitle MyDialogBox -InitialDirectory c:\temp -AllowMultiSelect -Filter "selected extensions (*.txt, *.log) | *.txt;*.log"
   C:\Temp\IPAddresses.txt
   C:\Temp\log.log

   Choose multiple file with different extensions
   Nota :It's important to have no white space in the extension name if you want to show them

.EXAMPLE
 Get-Help Get-FileName -Full

.INPUTS
   System.String
   System.Management.Automation.SwitchParameter

.OUTPUTS
   System.String

.NOTESs
  Version         : 1.0
  Author          : O. FERRIERE
  Creation Date   : 11/09/2019
  Purpose/Change  : Initial development

  Based on different pages :
   mainly based on https://blog.danskingdom.com/powershell-multi-line-input-box-dialog-open-file-dialog-folder-browser-dialog-input-box-and-message-box/
   https://code.adonline.id.au/folder-file-browser-dialogues-powershell/
   https://thomasrayner.ca/open-file-dialog-box-in-powershell/
#>
    [CmdletBinding()]
    [OutputType([string])]
    Param
    (
        # WindowsTitle help description
        [Parameter(
            Mandatory = $true,
            ValueFromPipelineByPropertyName = $true,
            HelpMessage = "Message Box Title",
            Position = 0)]
        [String]$WindowTitle,

        # InitialDirectory help description
        [Parameter(
            Mandatory = $true,
            ValueFromPipelineByPropertyName = $true,
            HelpMessage = "Initial Directory for browsing",
            Position = 1)]
        [String]$InitialDirectory,

        # Filter help description
        [Parameter(
            Mandatory = $false,
            ValueFromPipelineByPropertyName = $true,
            HelpMessage = "Filter to apply",
            Position = 2)]
        [String]$Filter = "All files (*.*)|*.*",

        # AllowMultiSelect help description
        [Parameter(
            Mandatory = $false,
            ValueFromPipelineByPropertyName = $true,
            HelpMessage = "Allow multi files selection",
            Position = 3)]
        [Switch]$AllowMultiSelect
    )

    # Load Assembly
    Add-Type -AssemblyName System.Windows.Forms

    # Open Class
    $OpenFileDialog = New-Object System.Windows.Forms.OpenFileDialog

    # Define Title
    $OpenFileDialog.Title = $WindowTitle

    # Define Initial Directory
    if (-Not [String]::IsNullOrWhiteSpace($InitialDirectory))
    {
        $OpenFileDialog.InitialDirectory = $InitialDirectory
    }

    # Define Filter
    $OpenFileDialog.Filter = $Filter

    # Check If Multi-select if used
    if ($AllowMultiSelect)
    {
        $OpenFileDialog.MultiSelect = $true
    }
    $OpenFileDialog.ShowHelp = $true    # Without this line the ShowDialog() function may hang depending on system configuration and running from console vs. ISE.
    $OpenFileDialog.ShowDialog() | Out-Null
    if ($AllowMultiSelect)
    {
        return $OpenFileDialog.Filenames
    }
    else
    {
        return $OpenFileDialog.Filename
    }
}

# =========================== import ===========================================

# select CSV file
if ($args[0]) {
  $fileName = $args[0]
} else {
  $fileName = Get-FileName -WindowTitle "Select file" -InitialDirectory . -Filter "Excel (*.xlsx) | *.xlsx"
}
if ($fileName -eq "") {
  Return
}

$fileNameCSV = [System.IO.Path]::GetFileNameWithoutExtension($fileName) + ".csv"
$fileNameOut = [System.IO.Path]::GetFileNameWithoutExtension($fileName) + "_out.csv"
#REMOVED filename must have currency at the end, eg. xStation5_cashOperations_123456789_01-01-2024--30-12-2024_EUR.csv
#REMOVED $fileCurrency = $fileName.Substring($fileName.lastIndexOf('_') + 1, 3)
$fileCurrency = (Import-Excel -Path $fileName -WorksheetName "CASH OPERATION HISTORY" -NoHeader -StartRow 6).P6[0]

# convert Excel to CSV
(Import-Excel -Path $fileName -WorksheetName "CASH OPERATION HISTORY" -StartRow 11 -StartColumn 2 -AsDate "Time") | Export-Csv $fileNameCSV -Delimiter ";" -Encoding UTF8 -NoTypeInformation
# remove last row (sum)
(Get-Content $fileNameCSV) | Select-Object -SkipLast 1 | Set-Content $fileNameCSV -Encoding UTF8

# import
Import-Csv -Path $fileNameCSV -Delimiter ";" -Encoding UTF8 |
# rename columns
Select-Object ID,Type,@{expression={$_.Time}; label='DateTime'},@{expression={$_.Symbol}; label='Ticker symbol'},@{expression={$_.Comment}; label='Note'},@{expression={$_.Amount}; label='Value'} |
foreach {
  # skip line (duplicity)
  if ($_.Type.Contains("close trade")) {
    return
  }

  # add transaction currency
  $_ | Add-Member -MemberType NoteProperty -Name "Transaction Currency" -Value $fileCurrency
  # add date
  $_ | Add-Member -MemberType NoteProperty -Name Date -Value $_.DateTime.split(" ")[0]
  # add time
  $_ | Add-Member -MemberType NoteProperty -Name Time -Value $_.DateTime.split(" ")[1]

  # add quantity and price
  if ($_.Note.Contains("OPEN BUY") -or $_.Note.Contains("CLOSE BUY")) {
    if ($_.Note.Contains("/")) {
      $index1 = $_.Note.IndexOf("/")
      $index2 = $_.Note.IndexOf("@")
    } else {
      $index1 = $index2 = $_.Note.IndexOf("@")
    }
    $_ | Add-Member -MemberType NoteProperty -Name Shares -Value $_.Note.Substring(9, $index1 - 9).Trim()
    $_ | Add-Member -MemberType NoteProperty -Name "Gross Amount" -Value $_.Note.Substring($index2 + 2).Trim()
    
    #if ($_.Note.Contains("CLOSE BUY")) {
      $_.Value = [double]$_.Shares * [double]$_."Gross Amount"
    #}
  } else {
    $_ | Add-Member -MemberType NoteProperty -Name Shares -Value ""
    $_ | Add-Member -MemberType NoteProperty -Name "Gross Amount" -Value ""
  }
  # add gross amount currency
  $_ | Add-Member -MemberType NoteProperty -Name "Currency Gross Amount" -Value $fileCurrency

  # add cash account (EUR, USD)
  $_ | Add-Member -MemberType NoteProperty -Name "Cash Account" -Value $fileCurrency
  # add securities account (XTB EUR, XTB USD ...)
  $_ | Add-Member -MemberType NoteProperty -Name "Securities Account" -Value "XTB $fileCurrency"

  # export item
  $_
} |
Export-Csv $fileNameOut -Delimiter ";" -Encoding UTF8 -NoTypeInformation
#cleanup
Remove-Item $fileNameCSV